options(width = 120)

#*******************************************************************************#
#* Program: 1_harmonzd.tus_cps.1992.through.2019.r                              #
#* Date: August 2024                                                            #
#*                                                                              #
#* Purpose: R program to create permanent R dataset from the 1992-2019          #
#*          harmonized data file.                                               #
#*                                                                              #
#*   Notes: The program uses the package readr.                                 #
#*          The working directory location need to be modified                  #
#*          to match location of the data files and the format of the levels    #
#*          of the variables.                                                   #
#*          This program uses the formats in the file                           #
#*          "harmonzd.tus_cps.1992.through.2019.format_levels.r"                #
#*          Formatted variables with level definitions are created as factors.  #
#*                                                                              #
#* Modifications: As of 8/2024, this program replaces                           #
#*                "5_harmonzd_tus_cps_1992_through_2019_beta" which was         #
#*                previously available on the TUS-CPS Website. The updated      #
#*                set of programs adds variable formats.                        #
#*******************************************************************************#

##################################################################################
# Load libraries
#
library(readr)

##################################################################################
# Set working directory
##################################################################################

setwd("")
getwd()

##################################################################################
## Create data dictionary with variable names, widths, type and labels
##################################################################################


data_diccionary <-
'SurWave     1   2     n   "Survey wave" 
SurYear     3   6     n    "Survey year"
SurMonth    7   8     n    "Survey month" 
SpanHome    9   10    n    "Is Spanish the only language spoken by all members of this household who are 15 years of age or older" 
HMOwnRnt    11  11    n    "Are your living quarters..." 
FamInc      12  13    n    "Family income" 
HHNum       14  15    n    "Number of persons in household"
OwnBsFrm    16  17    n    "Does anyone in this household have a business or a farm?" 
Region      18  18    n    "Region" 
FIPS        19  20    n    "State (FIPS Code)" 
State       21  40    c    "State" 
StCounty    41  45    n    "State/County" 
MetStat     46  46    n    "Metropolitan status" 
Age         47  48    n    "Age" 
Sex         49  49    n    "Sex" 
AFEver      50  51    n    "Did you ever serve on active duty in the U.S. Armed Forces" 
Race92      52  52    n    "Race (1992-2002)" 
Race03      53  54    n    "Race (2003 and later)" 
Hispan      55  55    n    "Hispanic or non-Hispanic" 
Marital     56  57    n    "Marital status" 
Edu         58  59    n    "Highest level of school completed or degree received" 
EmplStat    60  61    n    "Employment status" 
Natvty      62  64    n    "Country of birth" 
NatvtyM     65  67    n    "Mother-s country of birth" 
NatvtyF     68  70    n    "Father-s Country of birth" 
Citizen     71  71    n    "Citizenship status" 
YEARINUS    72  74    n    "Years in U.S." 
AgeInUS     75  77    n    "Immigrant-s age of entry" 
ABSNTRSN    78  79    n    "What is the main reason you were absent from work last week" 
ABSNTPAY    80  81    n    "Are you being paid by your employer for any of the time off last week" 
MULTJOBS    82  83    n    "Do you have more than one job" 
OTWk        84  85    n    "Last week, did you work any overtime or extra hours (at your main job) that you do not usually work" 
OTWkAMT     86  91    n    "Calculated weekly overtime amount"
CHILDNUM    92  93    n    "Number of children <18 years of age"
CHILD       94  95    n    "Presence of own children <18 years of age by selected age group" 
CHILD2      96  97    n    "Any own children 0-2 years of age" 
CHILD5      98  99    n    "Any own children 3-5 years of age" 
CHILD13     100 101   n    "Any own children 6-13 years of age" 
CHILD17     102 103   n    "Any own children 14-17 years of age" 
WRKCLSS1    104 105   n    "Individual class of worker code on first job" 
WRKCLSS2    106 107   n    "Individual class of worker code on second job" 
IND1DT95    108 109   n    "Detailed industry recode - job 1 (1995-2002)" 
IND1DT03    110 111   n    "Detailed industry recode - job 1 (2003 and later)" 
IND2DT95    112 113   n    "Detailed industry recode - job 2 (1995-2002)" 
IND2DT03    114 115   n    "Detailed industry recode - job 2 (2003 and later)" 
OCC1DT95    116 117   n    "Detailed occupation recode - job 1 (1995-2002)" 
OCC1DT03    118 119   n    "Detailed occupation recode - job 1 (2003 and later)" 
OCC2DT95    120 121   n    "Detailed occupation recode - job 2 (1995-2002)" 
OCC2DT03    122 123   n    "Detailed occupation recode - job 2 (2003 and later)" 
IND1MJ95    124 125   n    "Major industry recode - job 1 (1995-2002)" 
IND1MJ03    126 127   n    "Major industry recode - job 1 (2003 and later)" 
IND2MJ95    128 129   n    "Major industry recode - job 2 (1995-2002)" 
IND2MJ03    130 131   n    "Major industry recode - job 2 (2003 and later)" 
OCC1MJ95    132 133   n    "Major occupation recode - job 1 (1995-2002)" 
OCC1MJ03    134 135   n    "Major occupation recode - job 1 (2003 and later)" 
OCC2MJ95    136 137   n    "Major occupation recode - job 2 (1995-2002)" 
OCC2MJ03    138 139   n    "Major occupation recode - job 2 (2003 and later)" 
RecordID    140 161   c    "Record ID (Unique within survey month)"
QSTNUM      162 166   n    "Unique household identifier (Unique within survey month)"
OCCURNUM    167 168   n    "Unique person identifier (Unique within survey month)"
Cig100      181 182   n    "Smoked 100 cigarettes in entire life" 
CigAge      183 184   n    "Age first started smoking cigarettes fairly regularly" 
CigAgeX     185 186   n    "Age first started smoking cigarettes (never smoked regularly)" 
CGSTCRY     187 189   n    "State or country lived when first smoked part or all of a cigarette" 
CGSTCRYX    190 192   n    "State or country lived when first smoked part or all of a cigarette (never smoked regularly)" 
CIGNOW      193 194   n    "Now smoke cigarettes every day, some days or not at all" 
CIGSTAT     195 196   n    "Current cigarette smoking status" 
CPDD        197 198   n    "Number of cigarettes smoke a day (every day smokers)" 
CPDD20      199 200   n    "On average, now smoke more or less than 20 cigarettes each day (every day smokers*)" 
CIGDAY      201 202   n    "On how many of the past 30 days did you smoke cigarettes (some day smokers)" 
CIGDAY20    203 204   n    "Would you say you smoked on at least 12 days in the past 30 days (some day smokers*)" 
CPDS        205 206   n    "On days smoked, number of cigarettes smoke a day (some day smokers)" 
CIGTYPE     207 208   n    "Type of cigarette (menthol/non-menthol, current smokers)" 
TTFC        209 212   n    "Time to first cigarette, minutes (current smokers)" 
TTFC30      213 214   n    "Smoke first cigarette of the day within first 30 minutes (current smokers)" 
BUYCIG      215 216   n    "Do you usually buy your own cigarettes? (current smokers)" 
BUYPKCR     217 218   n    "Do you usually buy your cigarettes by the pack or by the carton? (current smokers)" 
PRICELPK    219 222   n    "Price paid for last pack of cigarettes (current smokers)"
PRICELCR    223 227   n    "Price paid for last carton of cigarettes (current smokers)"
DISCNTS     228 229   n    "Did you use coupons, rebates, or any other special promotions when you bought your last (pack/carton) of cigarettes" 
CIGWHERE    230 231   n    "Where last (pack/carton) purchased (current smokers)" 
CIGSTATE    232 233   n    "State where purchased last (pack/carton) (current smokers)" 
CIGINDRS    234 235   n    "Buy last (pack/carton) of cigarettes from an Indian reservation (current smokers)" 
CIGOTWAY    236 237   n    "Other way purchased last (pack/carton) of cigarettes (current smokers)" 
SNGLCIG     238 239   n    "In the last 2 months, have you bought (or traded for) any single or individual cigarette (current smokers)" 
SNGLPRCE    240 243   n    "Price paid for last  single or individual cigarette (current smokers)"
SNGLWHER    244 245   n    "Where last single or individual cigarette purchased (current smokers)" 
SNGLSTCT    246 248   n    "State (or country) where purchased last single or individual cigarette (current smokers)" 
YRSSMKDC    249 250   n    "Number of years smoked every day (every day smokers)" 
YRSSMKDG    251 252   n    "For how long have you smoked every day (every day smokers)" 
YRSSMKDH    253 255   n    "Number of years smoked every day (harmonized, every day smokers)" 
MENTHREG    256 257   n    "Have you ever smoked menthol cigarettes for 6 months or more (current smokers)" 
MENTHGRP    258 259   n    "For how long (have you smoked/did you smoke) menthol cigarettes (current smokers)" 
MENTHOL     260 261   n    "Menthol use (constructed, current smokers" 
SDEVRDLY    262 263   n    "Ever smoked cigarettes every day for at least 6 months (some day smokers)" 
SDEVRCIG    264 265   n    "When you last smoked every day, on average how many cigarettes did you smoke daily (some day smokers)" 
SDYRSDLY    266 269   n    "Years since smoked every day (some day smokers)" 
SDYRSMKC    270 271   n    "Years smoked every day (some day smokers)" 
SDYRSMKG    272 273   n    "For how long have you smoked every day? (some day smokers)" 
CIGSTA12    274 275   n    "Around this time 12 months ago, were you smoking cigarettes every day, some days or not at all? (current smokers)" 
CPDD12      276 277   n    "Around this time 12 months ago, number of cigarettes smoked a day (current smoker, every day smokers 12 months ago)" 
CIGDAY12    278 279   n    "Around this time 12 months ago, number of the past 30 days smoked cigarettes (some day smokers 12 months ago)" 
CPDS12      280 281   n    "Around this time 12 months ago, number of cigarettes smoked a day (current smoker, some day smokers 12 months ago)" 
EVSTP1DD    282 283   n    "Have you ever stopped smoking for one day or longer because you were trying to quit smoking? (every day smokers)" 
PYSTP1DD    284 285   n    "During the past 12 months, have you stopped smoking for one day or longer because you were trying to quit smoking? (every day smokers)" 
XPYSTPDN    286 287   n    "How many times during the past 12 months have you stopped smoking for one day or longer because you were trying to quit smoking? (every day smokers)" 
XPYSTPDC    288 289   n    "How many times during the past 12 months have you stopped smoking for one day or longer because you were trying to quit smoking? (every day smokers)" 
PYMLTDU     290 294   n    "During the past 12 months, what is the longest length of time you stopped smoking because you were trying to quit smoking? (every day smokers, number of months, not truncated)" 
PYMLTD      295 298   n    "During the past 12 months, what is the longest length of time you stopped smoking because you were trying to quit smoking? (every day smokers, number of months)" 
EVTRYQD     299 300   n    "Have you ever made a serious attempt to stop smoking because you were trying to quit - even if you stopped for less than a day? (every day smokers)" 
PYTRYQD     301 302   n    "During the past 12 months, have you made a serious attempt to stop smoking because you were trying to quit <even if you stopped for less than a day>? (every day smokers)" 
EVSTP1DS    303 304   n    "Have you ever stopped smoking for one day or longer because you were trying to quit smoking? (some days smokers, 12+ days)" 
PYSTP1DS    305 306   n    "During the past 12 months, have you stopped smoking for one day or longer because you were trying to quit smoking? (some days smokers, 12+ days)" 
XPYSTPSN    307 310   n    "How many times during the past 12 months have you stopped smoking for one day or longer because you were trying to quit smoking? (some days smokers, 12+ days)" 
XPYSTPSC    311 312   n    "How many times during the past 12 months have you stopped smoking for one day or longer because you were trying to quit smoking? (some days smokers, 12+ days)" 
PYMLTHSU    313 317   n    "During the past 12 months, what is the longest length of time you stopped smoking because you were trying to quit smoking? (some days smokers, 12+ days, number of months, not truncated)" 
PYMLTHS     318 321   n    "During the past 12 months, what is the longest length of time you stopped smoking because you were trying to quit smoking? (some days smokers, 12+ days, number of months)" 
EVTRYQS     322 323   n    "Have you ever made a serious attempt to stop smoking because you were trying to quit - even if you stopped for less than a day? (some days smokers, 12+ days)" 
PYTRYQS     324 325   n    "During the past 12 months, have you made a serious attempt to stop smoking because you were trying to quit <even if you stopped for less than a day>? (some days smokers, 12+ days)" 
PYQUITSD    326 327   n    "During the past 12 months, have you tried to quit smoking completely? (some days smokers, < 12 days)" 
EVTRYQCS    328 329   n    "Have you ever tried to quit smoking completely? (some days smokers, < 12 days)" 
PYMLTLSU    330 335   n    "During the past 12 months, what is the longest length of time you stopped smoking because you were trying to quit smoking? (some days smokers, < 12 days, number of months, not truncated)" 
PYMLTLS     336 339   n    "During the past 12 months, what is the longest length of time you stopped smoking because you were trying to quit smoking? (some days smokers, < 12 days, number of months)" 
PYQTNRT1    340 341   n    "Thinking back to the (last time/time) you tried to quit smoking in the past 12 months. Did you use a nicotine patch?" 
PYQTNRT2    342 343   n    "Thinking back to the (last time/time) you tried to quit smoking in the past 12 months. Did you use a nicotine gum or nicotine lozenge?" 
PYQTNRT3    344 345   n    "Thinking back to the (last time/time) you tried to quit smoking in the past 12 months. Did you use a nicotine nasal spray or nicotine inhaler?" 
PYQTNRT4    346 347   n    "Thinking back to the (last time/time) you tried to quit smoking in the past 12 months. Did you use a prescription pill?" 
PYQTHLP1    348 349   n    "Thinking back to the (last time/time) you tried to quit smoking in the past 12 months. Did you use a telephone help line or quit line?" 
PYQTHLP2    350 351   n    "Thinking back to the (last time/time) you tried to quit smoking in the past 12 months. Did you use a stop smoking clinic, class or support group?" 
PYQTHLP3    352 353   n    "Thinking back to the (last time/time) you tried to quit smoking in the past 12 months. Did you use one-on-one counseling?" 
PYQTHLP4    354 355   n    "Thinking back to the (last time/time) you tried to quit smoking in the past 12 months. Did you use help or support from friends or family?" 
PYQTHLP5    356 357   n    "Thinking back to the (last time/time) you tried to quit smoking in the past 12 months. Did you use the internet or World Wide Web/web-based program?" 
PYQTHLP6    358 359   n    "Thinking back to the (last time/time) you tried to quit smoking in the past 12 months. Did you use books, pamphlets, videos or other materials?" 
PYQTHLP7    360 361   n    "Thinking back to the (last time/time) you tried to quit smoking in the past 12 months. Did you use acupuncture or hypnosis?" 
PYQHLP7A    362 363   n    "Thinking back to the (last time/time) you tried to quit smoking in the past 12 months. Which did you use acupuncture or hypnosis or both?" 
PYQTPTN1    364 365   n    "The (last time/time) you tried to quit smoking in the past 12 months. Did you try to quit by gradually cutting back on cigarettes?" 
PYQTPTN2    366 367   n    "The (last time/time) you tried to quit smoking in the past 12 months. Did you try to quit by switching to chewing tobacco, snuff, <snus>, cigars or pipes?" 
PYQTPTN3    368 369   n    "The (last time/time) you tried to quit smoking in the past 12 months. Did you switch to a lighter cigarette in order to try to quit?" 
PYQTPTN4    370 371   n    "The (last time/time) you tried to quit smoking in the past 12 months. Did you try to give up cigarettes all at once?" 
PYQPTN4A    372 373   n    "The (last time/time) you tried to quit smoking in the past 12 months. When you tried to quit all at once, would you say you tried to quit -cold turkey-" 
PYQTPTN5    374 375   n    "The (last time/time) you tried to quit smoking in the past 12 months. Did you try to quit by switching to electronic or e-cigarettes?" 
DOCSEEN     376 377   n    "In the past year/12 months have you seen a medical doctor?" 
DOCADVYR    378 379   n    "During the past year/12 months, did any medical doctor advise you to stop smoking?" 
DOCADVQL    380 381   n    "In the past 12 months, when a medical doctor advised you to quit smoking, did the doctor also suggest that you call or use a telephone help line or quit line?" 
DOCADVEV    382 383   n    "Has a medical doctor ever advised you to stop smoking?" 
DDSSEEN     384 385   n    "In the past year/12 months have you seen a dentist?" 
DDSADVYR    386 387   n    "During the past year/12 months, did any dentist advise you to stop smoking?" 
DDSADVQL    388 389   n    "In the past 12 months, when a dentist advised you to quit smoking, did the dentist also suggest that you call or use a telephone help line or quit line?" 
DDSADVEV    390 391   n    "Has a dentist ever advised you to stop smoking?" 
INTQT6MO    392 393   n    "Are you seriously considering stopping/quitting smoking within the next 6 months?" 
INTQT30D    394 395   n    "Are you planning to stop/quit within the next 30 days?" 
INTRQUIT    396 397   n    "Overall, on a scale from 1 to 10 where 1 is not at all interested and 10 is extremely interested, how interested are you in quitting smoking?" 
QUITSUCS    398 399   n    "If you did try to quit smoking altogether in the next 6 months, how likely do you think you would be to succeed..." 
FMEVRDLY    400 401   n    "Ever smoked cigarettes every day for at least 6 months (former smokers)" 
FMYRSDLY    402 406   n    "Years since smoked every day (former smokers who smoked every day for at least 6 months)" 
FMEVRCIG    407 408   n    "When you last smoked every day, on average how many cigarettes did you smoke daily? (former smokers who smoked every day for at least 6 months)" 
FMYRSMKC    409 410   n    "Years smoked every day (former smokers who smoked every day for at least 6 months)" 
FMYRSMKG    411 412   n    "Years smoked every day (former smokers who smoked every day for at least 6 months)" 
FMLOTQT     413 417   n    "About how long has it been since you completely stopped smoking cigarettes (years, former smokers)" 
FMLOTQTC    418 419   n    "About how long has it been since you completely stopped smoking cigarettes (categorical, former smokers)" 
FMCGST12    420 421   n    "Around this time 12 months ago, were you smoking cigarettes every day, some days or not at all? (former smokers who quit within the past year)" 
FMCPDD12    422 423   n    "Around this time 12 months ago, on the average, about how many cigarettes did you smoke each day? (former smokers who quit within the past year, every day smokers around 12 months ago)" 
FMCIGD12    424 425   n    "Around this time 12 months ago, on how many of 30 days in the month did you smoke cigarettes? (former smokers who quit within the past year, some day smokers around 12 months ago)" 
FMCPDS12    426 427   n    "On the average, on those days, how many cigarettes did you usually smoke each day? (former smokers who quit within the past year, some day smokers around 12 months ago)" 
FMMNTHUB    428 429   n    "Type of cigarette (menthol/non-menthol, former smokers who quit within the past 5 years)" 
FMMNTHRG    430 431   n    "Ever smoked menthol cigarettes (former smokers who quit within the past 5 years)" 
FMMNTHGP    432 433   n    "How long did you smoke menthol cigarettes (former smokers who quit within the past 5 years and smoked menthol)" 
FMQTNRT1    434 435   n    "When you quit smoking completely did you use a nicotine patch (former smokers who quit within the past 3 years)?" 
FMQTNRT2    436 437   n    "When you quit smoking completely did you use a nicotine gum or nicotine lozenge? (former smokers who quit within the past 3 years)" 
FMQTNRT3    438 439   n    "When you quit smoking completely did you use a nicotine nasal spray or nicotine inhaler? (former smokers who quit within the past 3 years)" 
FMQTNRT4    440 441   n    "When you quit smoking completely did you use a prescription pill? (former smokers who quit within the past 3 years)" 
FMQTHLP1    442 443   n    "When you quit smoking completely did you use a telephone help line or quit line? (former smokers who quit within the past 3 years)" 
FMQTHLP2    444 445   n    "When you quit smoking completely did you use a stop smoking clinic, class or support group? (former smokers who quit within the past 3 years)" 
FMQTHLP3    446 447   n    "When you quit smoking completely did you use one-on-one counseling? (former smokers who quit within the past 3 years)" 
FMQTHLP4    448 449   n    "When you quit smoking completely did you use help or support from friends or family? (former smokers who quit within the past 3 years)" 
FMQTHLP5    450 451   n    "When you quit smoking completely did you use the internet or World Wide Web/web-based program? (former smokers who quit within the past 3 years)" 
FMQTHLP6    452 453   n    "When you quit smoking completely did you use books, pamphlets, videos or other materials? (former smokers who quit within the past 3 years)" 
FMQTHLP7    454 455   n    "When you quit smoking completely did you use acupuncture or hypnosis? (former smokers who quit within the past 3 years)" 
FMQHLP7A    456 457   n    "When you quit smoking completely which did you use acupuncture or hypnosis or both? (former smokers who quit within the past 3 years)" 
FMQTPTN1    458 459   n    "When you quit smoking completely did you try to quit by gradually cutting back on cigarettes? (former smokers who quit within the past 3 years)" 
FMQTPTN2    460 461   n    "When you quit smoking completely did you try to quit by switching to chewing tobacco, snuff, <snus>, cigars or pipes? (former smokers who quit within the past 3 years)" 
FMQTPTN3    462 463   n    "When you quit smoking completely did you switch to a -lighter- cigarette in order to try to quit? (former smokers who quit within the past 3 years)" 
FMQTPTN4    464 465   n    "When you quit smoking completely did you try to give up cigarettes all at once? (former smokers who quit within the past 3 years)" 
FMQPTN4A    466 467   n    "When you quit smoking completely, when you tried to quit all at once, would you say you tried to quit -cold turkey-? (former smokers who quit within the past 3 years)" 
FMQTPTN5    468 469   n    "When you quit smoking completely did you try to quit by switching to electronic or e-cigarettes? (former smokers who quit within the past 3 years)" 
FMDOCSEE    470 471   n    "In the 12 months before you completely quit smoking did you see a medical doctor? (former smokers who quit within the past 3 years)" 
FMDOCADV    472 473   n    "During the 12 months before you completely quit smoking, did any medical doctor advise you to stop smoking? (former smokers who quit within the past 3 years)" 
FMTTFC      474 477   n    "During the year before your quit smoking, time to first cigarette, minutes (former smokers who quit within the past 3 years)" 
FMTTFC30    478 479   n    "During the year before your quit smoking, smoke first cigarette of the day within first 30 minutes (former smokers who quit within the past 3 years)" 
PIPEEVER    480 481   n    "Ever used pipes <regular pipe filled with tobacco>?" 
CIGREVER    482 483   n    "Ever used cigars <regular cigar, cigarillos or little filtered cigars>?" 
CHEWEVER    484 485   n    "Ever used chewing tobacco?" 
SNFFEVER    486 487   n    "Ever used snuff?" 
HOOKEVER    488 489   n    "Ever used a water or hookah pipe filled with tobacco?" 
SMKLEVER    490 491   n    "Ever used smokeless tobacco?" 
ECIGEVER    492 493   n    "Ever used e-cigarettes?" 
PIPENOW     494 495   n    "Now use pipes <regular pipe filled with tobacco>?" 
CIGRNOW     496 497   n    "Now use cigars <regular cigar, cigarillos or little filtered cigars>?" 
CHEWNOW     498 499   n    "Now use chewing tobacco?" 
SNFFNOW     500 501   n    "Now use snuff?" 
HOOKNOW     502 503   n    "Now use a water or hookah pipe filled with tobacco?" 
SMKLNOW     504 505   n    "Now use smokeless tobacco?"  
ECIGNOW     506 507   n    "Now use e-cigarettes?"  
PIPENOW2    508 509   n    "Now use pipes <regular pipe filled with tobacco>?" 
CIGRNOW2    510 511   n    "Now use cigars <regular cigar, cigarillos or little filtered cigars>?" 
CHEWNOW2    512 513   n    "Now use chewing tobacco?" 
SNFFNOW2    514 515   n    "Now use snuff?" 
HOOKNOW2    516 517   n    "Now use a water or hookah pipe filled with tobacco?" 
SMKLNOW2    518 519   n    "Now use smokeless tobacco?" 
ECIGNOW2    520 521   n    "Now use e-cigarettes?" 
PIPECURR    522 523   n    "Current pipe <regular pipe filled with tobacco> use status" 
CIGRCURR    524 525   n    "Current cigar <regular cigar, cigarillos or little filtered cigars> use status" 
CHEWCURR    526 527   n    "Current chewing tobacco use status" 
SNFFCURR    528 529   n    "Current snuff use status" 
HOOKCURR    530 531   n    "Current water or hookah pipe filled with tobacco use status" 
SMKLCURR    532 533   n    "Current smokeless tobacco use status" 
ECIGCURR    534 535   n    "Current e-cigarette use status" 
PIPESTAT    536 537   n    "Current pipe <regular pipe filled with tobacco> use status" 
CIGRSTAT    538 539   n    "Current cigar <regular cigar, cigarillos or little filtered cigars> use status" 
CHEWSTAT    540 541   n    "Current chewing tobacco use status" 
SNFFSTAT    542 543   n    "Current snuff use status" 
HOOKSTAT    544 545   n    "Current water or hookah pipe filled with tobacco use status" 
SMKLSTAT    546 547   n    "Current smokeless tobacco use status" 
ECIGSTAT    548 549   n    "Current e-cigarette use status" 
PIPEDAY     550 551   n    "On how many of the past 30 days did you use pipes? (some day users)" 
CIGRDAY     552 553   n    "On how many of the past 30 days did you use cigars? (some day users)" 
CHEWDAY     554 555   n    "On how many of the past 30 days did you use chewing tobacco? (some day users)" 
SNFFDAY     556 557   n    "On how many of the past 30 days did you use snuff? (some day users)" 
HOOKDAY     558 559   n    "On how many of the past 30 days did you use a water or hookah pipe filled with tobacco? (some day users)" 
SMKLDAY     560 561   n    "On how many of the past 30 days did you use smokeless tobacco? (some day users)" 
ECIGDAY     562 563   n    "On how many of the past 30 days did you use e-cigarettes? (some day users)" 
PIPELOTQ    564 567   n    "About how long has it been since you completely quit smoking a regular pipe filled with tobacco? (years, former pipe smokers)" 
CIGRLOTQ    568 571   n    "About how long has it been since you completely quit smoking cigars, cigarillos or little filtered cigars? (years, former cigar smokers)" 
HOOKLOTQ    572 575   n    "About how long has it been since you completely quit smoking a water or hookah pipe filled with tobacco? (years, former hookah smokers)" 
SMKLLOTQ    576 579   n    "About how long has it been since you completely quit using smokeless tobacco? (years, former smokeless users)" 
ECIGLOTQ    580 583   n    "About how long has it been since you completely quit using e-cigarettes? (years, former e-cigarette users)" 
PIPEYRS     584 585   n    "In total, how many years (have you smoked/did you smoke) a regular pipe filled with tobacco? (ever pipe smokers)" 
CIGRYRS     586 587   n    "In total, how many years (have you smoked/did you smoke) cigars or cigarillos or little filtered cigars? (ever cigar smokers)" 
HOOKYRS     588 589   n    "In total, how many years (have you smoked/did you smoke) a water or hookah pipe filled with tobacco? (ever hookah smokers)" 
SMKLYRS     590 591   n    "In total, how many years (have you used/did you use) smokeless tobacco? (ever smokeless users)" 
ECIGYRS     592 593   n    "In total, how many years (have you used/did you use) e-cigarettes? (ever e-cigarette users)" 
CIGRTYPE    594 595   n    "During the past 30 days, what type of cigar did you use most often?" 
CIGRFLVR    596 597   n    "When you smoke a cigar is it usually flavored?" 
PIPEFLVR    598 599   n    "When you smoke a regular pipe filled with tobacco is it usually flavored?" 
HOOKFLVR    600 601   n    "When you smoke a water/hookah pipe filled with tobacco is it usually flavored" 
SMKLFLVR    602 603   n    "When you use smokeless tobacco is it usually flavored" 
ECIGFLVR    604 605   n    "When you use an e-cigarette is it usually flavored?" 
ECIGBRDG    606 607   n    "Do you use e-cigarettes because you can use them at times when or in places where smoking cigarettes isn-t allowed?" 
ECIGTOQT    608 609   n    "Do you use e-cigarettes to help you quit smoking cigarettes?" 
FMECIGBR    610 611   n    "Did you use e-cigarettes because you could use them at times when or in places where smoking cigarettes wasn-t allowed?" 
FMECIGQT    612 613   n    "Did you use e-cigarettes to help you quit smoking cigarettes?" 
TTFOTP      614 617   n    "Time to first other tobacco product, minutes (current users)" 
TTFOTP30    618 619   n    "Use first other tobacco product of the day within first 30 minutes (current users)" 
FMTTFOTP    620 623   n    "In the 12 months before you quit, time to first other tobacco product, minutes (former users)" 
FMTTFO30    624 625   n    "In the 12 months before you quit, use first other tobacco product of the day within first 30 minutes (former users)" 
PYSTPOTP    626 627   n    "During the past 12 months, have you stopped using other tobacco product for one day or longer because you were trying to quit? (every day and some days >11 users)" 
PYLTOTP     628 631   n    "During the past 12 months, what is the longest length of time you stopped using other tobacco product because you were trying to quit? (every day and some days >11 users, number of months)" 
WKAREA92    632 633   n    "Which of these best describes the area in which you work most of the time? (1992-2002 only)" 
WKAREA03    634 635   n    "Which of these best describes the area in which you work most of the time? (2003-2007 only)" 
WKINDPLC    636 637   n    "Indoor place mainly work (2003-2007 only)" 
WKSTATE     638 639   n    "In which state, do you work on your main job or business (2003-2007 only)" 
WKAREA10    640 641   n    "Do you mainly work indoors or outdoors? (2010 and later)" 
WKINDPL1    642 643   n    "Indoor place mainly work (2010 and later)" 
WKSTATE1    644 645   n    "In which state do you work (2010 and later)" 
WKINDPL2    646 647   n    "Indoor place mainly work (2010 and later)" 
WKSTATE2    648 649   n    "In which state do you work on your main indoor job or business (2010 and later)" 
WKPOLICY    650 651   n    "Does your place of work have an official policy that restricts smoking in any way?" 
WKPOLIND    652 653   n    "Smoking policy for indoor public or common areas such as lobbies, rest rooms and lunch rooms" 
WKPOLWRK    654 655   n    "Smoking policy for work areas" 
WORKBAN     656 657   n    "Smoking policy at work (constructed)" 
WKSTPSMK    658 659   n    "Within the past 12 months, has your employer offered any stop smoking program or help to quit smoking" 
WKSMK2WK    660 661   n    "During the past two weeks has anyone smoked in the area in which you work?" 
ATSMKRST    662 663   n    "In restaurants, do you think that smoking should be:" 
ATSMKHSP    664 665   n    "In hospitals, do you think that smoking should be:" 
ATSMKIWA    666 667   n    "In indoor work areas, do you think that smoking should be:" 
ATSMKBAR    668 669   n    "In bars and cocktail lounges (and clubs), do you think that smoking should be:" 
ATSMKISP    670 671   n    "In indoor sporting events, do you think that smoking should be:" 
ATSMKISM    672 673   n    "In indoor shopping malls, do you think that smoking should be:" 
ATSMKPLY    674 675   n    "On outdoor children-s - playgrounds and sports fields, do you think that smoking should be:" 
ATSMKCAS    676 677   n    "Inside casinos, do you think that smoking should be:" 
ATSMCARO    678 679   n    "Inside a car, when there are other people present, do you think that smoking should..." 
ATSMCARC    680 681   n    "If children are present inside the car, do you think that smoking should..." 
HMSMKPOL    682 683   n    "Home smoking rules" 
MUHSMPOL    684 685   n    "In buildings with multiple apartments or living areas, do you think that smoking should be: " 
ATMINBUY    686 687   n    "In your opinion, how easy is it for minors to buy cigarettes and other tobacco products in your community?" 
ATSAMPLE    688 689   n    "Do you think that giving away free sample by tobacco companies should be:" 
ATTOBADV    690 691   n    "Do you think advertising of tobacco products should be:" 
SRWEIGHT    692 701   d     "Self Response Weight"'


##################################################################################
# Set location of TUS CPS 1992 through 2019 file
##################################################################################

Harmon_dat <- "./data/harmonzd.tus_cps.1992.through.2019.dat"

##################################################################################
# Create data dictionary
##################################################################################

tmp_file <- tempfile()
writeLines(data_diccionary, tmp_file)
data_diccionary_table <- read.table(tmp_file,sep="", col.names = c("variable", "initial_position","final_position", "type", "label"))

data_diccionary_tabletype <- as.character(data_diccionary_table$type)

### Create list for varaible name and type 
names(data_diccionary_tabletype) <-  data_diccionary_table$variable
data_diccionary_tabletype <- as.list(data_diccionary_tabletype)

##################################################################################
## Read TUS CPS 1992 through 2019 data
##


Harmon <- as.data.frame(read_fwf(Harmon_dat, 
                 col_positions = fwf_positions( data_diccionary_table$initial_position, 
                                 data_diccionary_table$final_position,
                                  col_names= data_diccionary_table$variable),
                 col_types = data_diccionary_tabletype  )) 

##################################################################################
## Attach labels
##################################################################################
for (i in 1:length(data_diccionary_table$variable))
{
  attr(Harmon[,i],"label") <- data_diccionary_table$label[i]
}
dim(Harmon)

##################################################################################
### Recode variables
Harmon$YEARINUS <- ifelse(!is.na(Harmon$YEARINUS), 
                          ifelse(Harmon$YEARINUS!=-1, Harmon$YEARINUS /10, Harmon$YEARINUS),
                                                      Harmon$YEARINUS)
Harmon$AgeInUS <- ifelse(!is.na(Harmon$AgeInUS), 
                          ifelse(!(Harmon$AgeInUS %in% c(-1,-9)), Harmon$AgeInUS /10, Harmon$AgeInUS),
                                                                  Harmon$AgeInUS)
table(Harmon$AgeInUS, useNA ="ifany")

Harmon$OTWkAMT <- ifelse(!is.na(Harmon$OTWkAMT), 
                          ifelse(Harmon$OTWkAMT!=-1, Harmon$OTWkAMT /100, Harmon$OTWkAMT),
                                                     Harmon$OTWkAMT)
Harmon$PRICELPK <- ifelse(!is.na(Harmon$PRICELPK), 
                          ifelse(!(Harmon$PRICELPK %in% c(-9,-3,-2,-1)), Harmon$PRICELPK /100, Harmon$PRICELPK),
                                                                         Harmon$PRICELPK)
Harmon$PRICELCR <- ifelse(!is.na(Harmon$PRICELCR), 
                          ifelse(!(Harmon$PRICELCR %in% c(-9,-3,-2,-1)), Harmon$PRICELCR /100, Harmon$PRICELCR),
                                                                         Harmon$PRICELCR)
Harmon$SNGLPRCE <- ifelse(!is.na(Harmon$SNGLPRCE), 
                          ifelse(!(Harmon$SNGLPRCE %in% c(-9,-3,-2,-1)), Harmon$SNGLPRCE /100, Harmon$SNGLPRCE),
                                                                         Harmon$SNGLPRCE)
Harmon$SDYRSDLY <- ifelse(!is.na(Harmon$SDYRSDLY), 
                          ifelse(!(Harmon$SDYRSDLY %in% c(-9,-3,-2,-1)), Harmon$SDYRSDLY /100, Harmon$SDYRSDLY),
                                                                         Harmon$SDYRSDLY)
Harmon$FMYRSDLY <- ifelse(!is.na(Harmon$FMYRSDLY), 
                          ifelse(!(Harmon$FMYRSDLY %in% c(-9,-3,-2,-1)), Harmon$FMYRSDLY /100, Harmon$FMYRSDLY),
                                                                         Harmon$FMYRSDLY)
Harmon$FMLOTQT <- ifelse(!is.na(Harmon$FMLOTQT), 
                          ifelse(!(Harmon$FMLOTQT %in% c(-9,-3,-2,-1)),  Harmon$FMLOTQT /100, Harmon$FMLOTQT),
                                                                         Harmon$FMLOTQT)
Harmon$XPYSTPSN <- ifelse(!is.na(Harmon$XPYSTPSN), 
                          ifelse(!(Harmon$XPYSTPSN %in% c(-9,-3,-2,-1)), Harmon$XPYSTPSN /100, Harmon$XPYSTPSN),
                                                                         Harmon$XPYSTPSN)
Harmon$PYMLTHS <- ifelse(!is.na(Harmon$PYMLTHS), 
                          ifelse(!(Harmon$PYMLTHS %in% c(-9,-3,-2,-1)),  Harmon$PYMLTHS /100, Harmon$PYMLTHS),
                                                                         Harmon$PYMLTHS)
Harmon$PYMLTLSU <- ifelse(!is.na(Harmon$PYMLTLSU), 
                          ifelse(!(Harmon$PYMLTLSU %in% c(-9,-3,-2,-1)), Harmon$PYMLTLSU /100, Harmon$PYMLTLSU),
                                                                         Harmon$PYMLTLSU)
Harmon$PYMLTLS <- ifelse(!is.na(Harmon$PYMLTLS), 
                          ifelse(!(Harmon$PYMLTLS %in% c(-9,-3,-2,-1)),  Harmon$PYMLTLS /100, Harmon$PYMLTLS),
                                                                         Harmon$PYMLTLS)
Harmon$PYMLTHSU <- ifelse(!is.na(Harmon$PYMLTHSU), 
                          ifelse(!(Harmon$PYMLTHSU %in% c(-9,-3,-2,-1)), Harmon$PYMLTHSU /100, Harmon$PYMLTHSU),
                                                                         Harmon$PYMLTHSU)
Harmon$PYMLTD <- ifelse(!is.na(Harmon$PYMLTD), 
                          ifelse(!(Harmon$PYMLTD %in% c(-9,-3,-2,-1)),  Harmon$PYMLTD /100, Harmon$PYMLTD),
                                                                        Harmon$PYMLTD)
Harmon$PYMLTDU <- ifelse(!is.na(Harmon$PYMLTDU), 
                          ifelse(!(Harmon$PYMLTDU %in% c(-9,-3,-2,-1)), Harmon$PYMLTDU /100, Harmon$PYMLTDU),
                                                                        Harmon$PYMLTDU)
Harmon$CIGRLOTQ <- ifelse(!is.na(Harmon$CIGRLOTQ), 
                          ifelse(!(Harmon$CIGRLOTQ %in% c(-99,-9,-3,-2,-1)), Harmon$CIGRLOTQ /100, Harmon$CIGRLOTQ),
                                                                             Harmon$CIGRLOTQ)
Harmon$PIPELOTQ <- ifelse(!is.na(Harmon$PIPELOTQ), 
                          ifelse(!(Harmon$PIPELOTQ %in% c(-99,-9,-3,-2,-1)), Harmon$PIPELOTQ /100, Harmon$PIPELOTQ),
                                                                             Harmon$PIPELOTQ)
Harmon$HOOKLOTQ <- ifelse(!is.na(Harmon$HOOKLOTQ), 
                          ifelse(!(Harmon$HOOKLOTQ %in% c(-99,-9,-3,-2,-1)), Harmon$HOOKLOTQ /100, Harmon$HOOKLOTQ),
                                                                             Harmon$HOOKLOTQ)
Harmon$SMKLLOTQ <- ifelse(!is.na(Harmon$SMKLLOTQ), 
                          ifelse(!(Harmon$SMKLLOTQ %in% c(-99,-9,-3,-2,-1)), Harmon$SMKLLOTQ /100, Harmon$SMKLLOTQ),
                                                                             Harmon$SMKLLOTQ)
Harmon$ECIGLOTQ <- ifelse(!is.na(Harmon$ECIGLOTQ), 
                          ifelse(!(Harmon$ECIGLOTQ %in% c(-99,-9,-3,-2,-1)), Harmon$ECIGLOTQ /100, Harmon$ECIGLOTQ),
                                                                             Harmon$ECIGLOTQ)
Harmon$PYLTOTP <- ifelse(!is.na(Harmon$PYLTOTP), 
                          ifelse(!(Harmon$PYLTOTP %in% c(-99,-9,-3,-2,-1)), Harmon$PYLTOTP /100, Harmon$PYLTOTP),
                                                                            Harmon$PYLTOTP)
Harmon$SRWEIGHT <- Harmon$SRWEIGHT/ 10000


##################################################################################
# Attach levels of varaibles
source("./harmonzd.tus_cps.1992.through.2019.format_levels.r", echo=TRUE)

##################################################################################
## Save file Harmon as a RDS File
##################################################################################

str(Harmon, list.len=ncol(Harmon))

saveRDS(Harmon, file = "./data/Harmon.RDS")


